unit PhryListView;

interface

uses
  Classes, StdCtrls, ExtCtrls, Messages, SysUtils, Windows, Controls, Forms,
  Graphics, ImgList, AdvancedPanel;

type
  TSortOrder = (soNone, soAscending, soDescending);

{ TPhryListColumn }

  TPhryListColumn = class(TCollectionItem)
  private
    FText :string;
    FAlignment :TAlignment;
    FWidth :Integer;
    FRealWidth :Integer;
    FImageIndex :Integer;
    FOwnerDraw :Boolean;
    FPosition :Integer;
    FPressed :Boolean;
    procedure SetAlignment(Value :TAlignment);
    procedure SetText(Value :string);
    procedure SetWidth(Value :Integer);
    procedure SetRealWidth(Value :Integer);
    procedure SetPosition(Value :Integer);
    procedure SetImageIndex(Value :Integer);
    procedure SetOwnerDraw(Value :Boolean);
  public
    constructor Create(Collection :TCollection); override;
    destructor Destroy; override;
    property Position :Integer read FPosition write SetPosition;
    property RealWidth :Integer read FRealWidth write SetRealWidth;
    property Pressed :Boolean read FPressed write FPressed;
  published
    property Alignment :TAlignment read FAlignment write SetAlignment;
    property Width :Integer read FWidth write SetWidth;
    property Text :string read FText write SetText;
    property ImageIndex :Integer read FImageIndex write SetImageIndex;
    property OwnerDraw :Boolean read FOwnerDraw write SetOwnerDraw;
  end;

{ TPhryListColumns }

  TPhryListColumns = class(TOwnedCollection)
  private
    function  GetItem(Index :Integer) :TPhryListColumn;
    procedure SetItem(Index :Integer; const Value :TPhryListColumn);
  protected
    procedure Update(Item :TCollectionItem); override;
  public
    function Add :TPhryListColumn;
    procedure Delete(Index :Integer);
    function Insert(Index :Integer) :TPhryListColumn;
    property Items[Index :Integer] :TPhryListColumn read GetItem write SetItem; default;
  end;

{ TPhryListItem }

  TPhryListItem = class(TCollectionItem)
  private
    FColor :TColor;
    FSelectedColor :TColor;
    FValues :TStrings;
    FImageIndex :Integer;
    FSelected :Boolean;
    procedure SetValues(Value :TStrings);
    procedure SetImageIndex(Value :Integer);
    procedure ValueChanged(Sender :TObject);
    procedure SetColor(Value :TColor);
    procedure SetSelectedColor(Value :TColor);
  public
    constructor Create(Collection :TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    property Selected :Boolean read FSelected write FSelected;
    property Index;
  published
    property Color :TColor read FColor write SetColor;
    property SelectedColor :TColor read FSelectedColor write SetSelectedColor;
    property Values :TStrings read FValues write SetValues;
    property ImageIndex :Integer read FImageIndex write SetImageIndex;
  end;

{ TPhryListItems }

  TPhryListItems = class(TOwnedCollection)
  private
    FSortList :TList;
    FSortColumn :Integer;
    FSortOrder :TSortOrder;
    function  GetItem(Index :Integer) :TPhryListItem;
    procedure SetItem(Index :Integer; const Value :TPhryListItem);
    procedure SetSortColumn(Value :Integer);
    procedure SetSortOrder(Value :TSortOrder);
  protected
    procedure Update(Item :TCollectionItem); override;
  public
    constructor Create(AOwner: TPersistent; ItemClass: TCollectionItemClass);
    procedure Sort;
    procedure InitItem(Item :TPhryListItem);
    function Add :TPhryListItem;
    procedure Delete(Index :Integer);
    function Insert(Index :Integer) :TPhryListItem;
    function IndexOf(Value :TPhryListItem) :Integer;
    property Items[Index :Integer] :TPhryListItem read GetItem write SetItem; default;
    property SortColumn :Integer read FSortColumn write SetSortColumn;
    property SortOrder :TSortOrder read FSortOrder write SetSortOrder;
  end;

{ TPhryListView }

  TPhryListView = class;
  TPLVInfoTipEvent = procedure(Sender: TObject;
                               Item :TPhryListItem;
                               Column :Integer;
                               var InfoTip: string) of object;

  TPLVDrawItemEvent = procedure(PaintCanvas :TCanvas;
                                Item :TPhryListItem;
                                Column :Integer;
                                PaintRect :TRect) of object;

  TPLVPopupEvent = procedure(Sender :TObject;
                             Item :TPhryListItem;
                             Column :Integer;
                             var Handled :Boolean) of object;

  TMouseMode = (mmNone, mmClicking, mmSplitting);

  TSplitMode = (smNone, smLeft, smRight);

  TPhryListView = class(TPaintBox)
  private
    { Private declarations }
    FColumns :TPhryListColumns;
    FLineHeight :Integer;
    FListItems: TPhryListItems;
    FOnInfoTip :TPLVInfoTipEvent;
    FOnPopup :TPLVPopupEvent;
    FTransparent :Boolean;
    FVScrollBar :TScrollBar;
    FItemOffset :Integer;
    FItemsShown :Integer;
    FImages :TCustomImageList;
    FOnDrawItem :TPLVDrawItemEvent;
    FImageBitmap :TBitmap;
    FClientWidth :Integer;
    FClientHeight :Integer;
    FClickedItem :Integer;
    FClickedColumn :Integer;
    FUpDownImage :TBitmap;
    FMouseMode :TMouseMode;
    FSplitMode :TSplitMode;
    FPrevX :Integer;
    FLastSelected :Integer;
    FTempBitmap :TBitmap;
    FWallpaper : TPanelWallpaper;
    FShowHeaders :Boolean;
    FAutoLineHeight :Boolean;
    procedure SetShowHeaders(Value :Boolean);
    procedure SetAutoLineHeight(Value :Boolean);
    procedure SetLineHeight(Value :Integer);
    procedure SetWallpaper(Value: TPanelWallpaper);
    procedure CMHintShow(var Message: TMessage); message CM_HINTSHOW;
    procedure SetItems(Value: TPhryListItems);
    procedure SetColumns(Value: TPhryListColumns);
    procedure SetImages(Value :TCustomImageList);
    function ItemRect(item, column :Integer) :TRect;
    procedure FindPos(Pos :TPoint; var item, column :Integer);
    procedure PaintItem(Item :TPhryListItem; Column :Integer);
    procedure Scrolled(Sender :TObject);
    procedure CalcColumnsWidth;
    procedure PaintHeader(ACanvas :TCanvas; Column :Integer);
    procedure HandleMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure HandleMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure HandleMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure HandleContextPopup(Sender :TObject; MousePos: TPoint; var Handled: Boolean);
  protected
    { Protected declarations }
    procedure PaintBorder(ACanvas :TCanvas; Rect: TRect; Raised :Boolean);
    procedure Paint; override;
    property OnMouseDown;
    property OnMouseUp;
    property OnMouseMove;
    property OnContextPopup;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
    property AutoLineHeight :Boolean read FAutoLineHeight write SetAutoLineHeight;
    property LineHeight :Integer read FLineHeight write SetLineHeight;
    property ShowHeaders :Boolean read FShowHeaders write SetShowHeaders;
    property Items :TPhryListItems read FListItems write SetItems;
    property Columns: TPhryListColumns read FColumns write SetColumns;
    property OnInfoTip :TPLVInfoTipEvent read FOnInfoTip write FOnInfoTip;
    property OnDrawItem :TPLVDrawItemEvent read FOnDrawItem write FOnDrawItem;
    property OnPopup :TPLVPopupEvent read FOnPopup write FOnPopup;
    property Transparent :Boolean read FTransparent write FTransparent;
    property Images :TCustomImageList read FImages write SetImages;
    property Wallpaper: TPanelWallpaper read FWallpaper write SetWallpaper;
  end;

procedure Register;

implementation

const
  cl3DHiLight  = COLOR_3DHILIGHT or $80000000;
  cl3DShadow   = COLOR_3DSHADOW or $80000000;
  LeftMargin   = 2;
  RightMargin  = 2;
  TopMargin :Integer = 2;
  BottomMargin = 2;
  MinColumnWidth = 10;
  bufimgupdown :Array[1..202] of byte = (
    66,77,202,0,0,0,0,0,0,0,118,0,0,0,40,0,
    0,0,17,0,0,0,7,0,0,0,1,0,4,0,0,0,
    0,0,84,0,0,0,196,14,0,0,196,14,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,128,0,0,128,
    0,0,0,128,128,0,128,0,0,0,128,0,128,0,128,128,
    0,0,192,192,192,0,128,128,128,0,0,0,255,0,0,255,
    0,0,0,255,255,0,255,0,0,0,255,0,255,0,255,255,
    0,0,255,255,255,0,119,120,247,119,127,255,255,255,240,0,
    0,0,119,135,127,119,120,119,119,119,240,0,0,0,119,135,
    127,119,119,135,119,127,112,0,0,0,120,119,119,247,119,135,
    119,127,112,0,0,0,120,119,119,247,119,120,119,247,112,0,
    0,0,135,119,119,127,119,120,119,247,112,0,0,0,136,136,
    136,143,119,119,143,119,112,0,0,0);
var
  ListSortColumn :Integer;
  ListSortOrder :TSortOrder;

{ TPhryListColumn }

constructor TPhryListColumn.Create(Collection: TCollection);
begin
  FWidth := MinColumnWidth;
  FImageIndex := -1;
  FOwnerDraw := False;
  FText := '';
  FAlignment := taLeftJustify;
  FPressed := False;
  inherited Create(Collection);
end;

destructor TPhryListColumn.Destroy;
begin
  inherited Destroy;
end;

procedure TPhryListColumn.SetAlignment(Value :TAlignment);
begin
  if Value <> FAlignment then
  begin
    FAlignment := Value;
  end;
end;

procedure TPhryListColumn.SetText(Value :string);
begin
  if Value <> FText then
  begin
    FText := Value;
  end;
end;

procedure TPhryListColumn.SetWidth(Value :Integer);
begin
  if (Value <> FWidth) and (Value >= MinColumnWidth) then
  begin
    FWidth := Value;
  end;
end;

procedure TPhryListColumn.SetRealWidth(Value :Integer);
begin
  if Value <> FRealWidth then
  begin
    FRealWidth := Value;
  end;
end;

procedure TPhryListColumn.SetPosition(Value :Integer);
begin
  if (Value <> FPosition) and (Value >= 0) then
  begin
    FPosition := Value;
  end;
end;

procedure TPhryListColumn.SetImageIndex(Value :Integer);
begin
  if Value <> FImageIndex then
  begin
    FImageIndex := Value;
  end;
end;

procedure TPhryListColumn.SetOwnerDraw(Value :Boolean);
begin
  if FOwnerDraw <> Value then
  begin
    FOwnerDraw := Value;
  end;
end;


{ TPhryListColumns }

procedure TPhryListColumns.Update(Item: TCollectionItem);
var
  i :Integer;
  ListView :TPhryListView;
begin
  ListView := TPhryListView(GetOwner);
  for i := 0 to ListView.Items.Count - 1 do
  begin
    while ListView.Items[i].Values.Count < Count do
      ListView.Items[i].Values.Add('');
    while ListView.Items[i].Values.Count > Count do
      ListView.Items[i].Values.Delete(Count);
  end;
  ListView.Refresh;
end;

function TPhryListColumns.Add: TPhryListColumn;
begin
  Result := TPhryListColumn(inherited Add);
end;

function TPhryListColumns.GetItem(Index: Integer): TPhryListColumn;
begin
  Result := TPhryListColumn(inherited GetItem(Index));
end;

procedure TPhryListColumns.SetItem(Index: Integer; const Value: TPhryListColumn);
begin
  inherited SetItem(Index, Value);
  Update(nil);
end;

procedure TPhryListColumns.Delete(Index: Integer);
begin
  Items[Index].Free;
  Changed;
end;

function TPhryListColumns.Insert(Index: Integer): TPhryListColumn;
begin
  Result := TPhryListColumn(inherited Insert(Index));
end;


{ TPhryListItem }

constructor TPhryListItem.Create(Collection: TCollection);
begin
  FValues := TStringList.Create;
  FSelected := False;
  FImageIndex := -1;
  FColor := clWindowText;
  FSelectedColor := clHighlightText;
  if Collection <> nil then
  begin
    TPhryListItems(Collection).InitItem(Self);
    TStringList(FValues).OnChange := ValueChanged;
  end;
  inherited Create(Collection);
end;

destructor TPhryListItem.Destroy;
begin
  FValues.Free;
  inherited Destroy;
end;

procedure TPhryListItem.Assign(Source: TPersistent);
begin
  if Source is TPhryListItem then
    with Source as TPhryListItem do
    begin
      Self.Values := Values;
      Self.ImageIndex := ImageIndex;
      Self.Selected := Selected;
      Self.Color := Color;
      Self.SelectedColor := SelectedColor;
    end
  else inherited Assign(Source);
end;

procedure TPhryListItem.SetValues(Value :TStrings);
begin
  if Value <> nil then FValues.Assign(Value);
end;

procedure TPhryListItem.SetImageIndex(Value :Integer);
begin
  if Value <> FImageIndex then
  begin
    FImageIndex := Value;
    Changed(True);
  end;
end;

procedure TPhryListItem.SetColor(Value :TColor);
begin
  if Value <> FColor then
  begin
    FColor := Value;
    if not Selected then
      Changed(true);
  end;
end;

procedure TPhryListItem.SetSelectedColor(Value :TColor);
begin
  if Value <> FSelectedColor then
  begin
    FSelectedColor := Value;
    if Selected then
      Changed(true);
  end;
end;

procedure TPhryListItem.ValueChanged(Sender :TObject);
begin
  Changed(True);
end;


{ TPhryListItems }

constructor TPhryListItems.Create(AOwner: TPersistent; ItemClass: TCollectionItemClass);
begin
  inherited Create(AOwner, ItemClass);
  FSortList := TList.Create;
  FSortColumn := 0;
  FSortOrder := soAscending;
end;

procedure TPhryListItems.SetSortColumn(Value :Integer);
begin
  if (Value >= 0) and (Value < TPhryListView(GetOwner).Columns.Count) then
    if Value <> FSortColumn then
    begin
      FSortColumn := Value;
      Sort;
    end;
end;

procedure TPhryListItems.SetSortOrder(Value :TSortOrder);
begin
  if Value <> FSortOrder then
  begin
    FSortOrder := Value;
    Sort;
  end;
end;

function CompareItems(Item1, Item2 :Pointer) :Integer;
var
  I1, I2 :TPhryListItem;
begin
  I1 := Item1;
  I2 := Item2;
  if I1.Values[ListSortColumn] < I2.Values[ListSortColumn] then
    Result := -1
  else if I1.Values[ListSortColumn] > I2.Values[ListSortColumn] then
    Result := 1
  else
    Result := 0;
  if ListSortOrder = soDescending then
    Result := Result * -1;
end;

procedure TPhryListItems.Sort;
var
  i :integer;
  item :TPhryListItem;
begin
  if (TPhryListView(GetOwner).Columns.Count > 0) and (SortOrder <> soNone) then
  begin
    BeginUpdate;
    FSortList.Clear;
    for i := Count - 1 downto 0 do
    begin
      item := TPhryListItem.Create(nil);
      item.Assign(Items[i]);
      FSortList.Add(item);
    end;
    Clear;
    ListSortColumn := SortColumn;
    ListSortOrder := SortOrder;
    FSortList.Sort(CompareItems);
    for i := 0 to FSortList.Count - 1 do
    begin
      item := Add;
      item.Assign(FSortList.Items[i]);
    end;
    FSortList.Clear;
    EndUpdate;
  end;
end;

procedure TPhryListItems.InitItem(Item :TPhryListItem);
var
  ListView :TPhryListView;
begin
  ListView := TPhryListView(GetOwner);
  if Item <> nil then
  begin
    while Item.Values.Count < ListView.Columns.Count do
      Item.Values.Add('');
    while Item.Values.Count > ListView.Columns.Count do
      Item.Values.Delete(ListView.Columns.Count);
  end;
end;

procedure TPhryListItems.Update(Item: TCollectionItem);
var
  ListView :TPhryListView;
begin
  ListView := TPhryListView(GetOwner);
  ListView.Refresh;
end;

function TPhryListItems.Add: TPhryListItem;
begin
  Result := TPhryListItem(inherited Add);
end;

function TPhryListItems.GetItem(Index: Integer): TPhryListItem;
begin
  Result := TPhryListItem(inherited GetItem(Index));
end;

procedure TPhryListItems.SetItem(Index: Integer; const Value: TPhryListItem);
begin
  inherited SetItem(Index, Value);
  Changed;
end;

procedure TPhryListItems.Delete(Index: Integer);
begin
  Items[Index].Free;
  Changed;
end;

function TPhryListItems.Insert(Index: Integer): TPhryListItem;
begin
  Result := TPhryListItem(inherited Insert(Index));
end;

function TPhryListItems.IndexOf(Value :TPhryListItem) :Integer;
var
  i :integer;
begin
  Result := -1;
  i := 0;
  while (i < Count) and (Result = -1) do
    if Items[i] = Value then
      Result := i;
end;


{ TPhryListView }

constructor TPhryListView.Create(AOwner :TComponent);
var
  s :TMemoryStream;
begin
  inherited Create(AOwner);
  Width := 250;
  Height := 150;
  ParentColor := False;
  FListItems := TPhryListItems.Create(Self, TPhryListItem);
  FColumns := TPhryListColumns.Create(Self, TPhryListColumn);
  FVScrollBar := TScrollBar.Create(Self);
  FVScrollBar.Kind := sbVertical;
  FVScrollBar.Min := 1;
  FVScrollBar.Max := 5;
  FVScrollBar.PageSize := 0;
  FVScrollBar.TabStop := False;
  FVScrollBar.Parent := Self.Parent;
  FVScrollBar.Visible := True;
  FVScrollBar.Ctl3d := False;
  FVScrollBar.ParentCtl3D := False;
  FVScrollBar.OnChange := Scrolled;
  FItemOffset := 0;
  FImageBitmap := TBitmap.Create;
  FImageBitmap.Canvas.Brush.Color := $00030303;
  FImageBitmap.TransparentColor := $00030303;
  FImageBitmap.TransparentMode := tmFixed;
  FImageBitmap.Transparent := True;
  FTempBitmap := TBitmap.Create;
  FClickedItem := -1;
  FClickedColumn := -1;
  OnMouseDown := HandleMouseDown;
  OnMouseUp := HandleMouseUp;
  OnMouseMove := HandleMouseMove;
  OnContextPopup := HandleContextPopup;
  FUpDownImage := TBitmap.Create;
  s := TMemoryStream.Create;
  s.Write(bufImgUpDown, sizeof(bufImgUpDown));
  s.Seek(0, soFromBeginning);
  FUpDownImage.LoadFromStream(s);
  s.Free;
  FMouseMode := mmNone;
  FSplitMode := smNone;
  FLastSelected := -1;
  FWallpaper := TPanelWallpaper.Create;
  FAutoLineHeight := True;
  FShowHeaders := True;
  FLineHeight := 12;
end;

destructor TPhryListView.Destroy;
begin
  FImageBitmap.Free;
  FTempBitmap.Free;
  FColumns.Free;
  FListItems.Free;
  FWallpaper.Free;
  inherited;
end;

procedure TPhryListView.SetWallpaper;
begin
  FWallpaper.Assign(Value);
end;

procedure TPhryListView.SetShowHeaders(Value :Boolean);
begin
  if Value <> FShowHeaders then
  begin
    FShowHeaders := Value;
    Refresh;
  end;
end;

procedure TPhryListView.SetAutoLineHeight(Value :Boolean);
begin
  if Value <> FAutoLineHeight then
  begin
    FAutoLineHeight := Value;
    if FAutoLineHeight then
      Refresh;
  end;
end;

procedure TPhryListView.SetLineHeight(Value :Integer);
begin
  if (Value <> FLineHeight) and (not FAutoLineHeight) and (LineHeight > 2) then
  begin
    FLineHeight := Value;
    Refresh;
  end;
end;

procedure TPhryListView.SetItems(Value: TPhryListItems);
begin
  FListItems.Assign(Value);
end;

procedure TPhryListView.SetColumns(Value: TPhryListColumns);
begin
  FColumns.Assign(Value);
end;

procedure TPhryListView.CMHintShow(var Message: TMessage);
var
  InfoTip :string;
  item, column :Integer;
begin
  if Assigned(FOnInfoTip) then
    with TCMHintShow(Message) do
    begin
      FindPos(HintInfo^.CursorPos, item, column);
      if (item < Items.Count) and (Item >= 0) then
        FOnInfoTip(Self, Items[item], column, InfoTip)
      else
        FOnInfoTip(Self, nil, column, InfoTip);
      with HintInfo^ do
      begin
        HintInfo.HintStr := InfoTip;
        HintInfo.CursorRect := ItemRect(item, column);
        HintInfo.HideTimeout := 120000;
        if (InfoTip = '') or (FMouseMode <> mmNone) then
          Message.Result := -1
        else
          Message.Result := 0;
      end;
    end
  else
    inherited;
end;

procedure TPhryListView.PaintBorder;
var
  TopColor   : TColor;
  BottomColor: TColor;

  procedure PaintRect(RectWidth: Integer);
  begin
    with ACanvas do begin
      while RectWidth > 0 do begin
        Pen.Color := TopColor;
        MoveTo(Rect.Left,Rect.Bottom-1);
        LineTo(Rect.Left,Rect.Top-1);
        MoveTo(Rect.Left,Rect.Top);
        Inc(Rect.Left);
        LineTo(Rect.Right,Rect.Top);
        MoveTo(Rect.Right-1,Rect.Top);
        Inc(Rect.Top);
        Pen.Color := BottomColor;
        LineTo(Rect.Right-1,Rect.Bottom);
        MoveTo(Rect.Right-1,Rect.Bottom-1);
        Dec(Rect.Right);
        LineTo(Rect.Left-2,Rect.Bottom-1);
        MoveTo(Rect.Left-1,Rect.Bottom-1);
        Dec(Rect.Bottom);
        Dec(RectWidth);
      end;
    end;
  end;

begin
  if Raised then
  begin
    ACanvas.Pen.Style := psSolid;
    ACanvas.Pen.Width := 1;
    TopColor := cl3DLight;
    BottomColor := cl3DDkShadow;
    PaintRect(1);
    TopColor := cl3DHiLight;
    BottomColor := cl3DShadow;
    PaintRect(1);
  end else
  begin
    ACanvas.Pen.Style := psSolid;
    ACanvas.Pen.Width := 1;
    TopColor := cl3DShadow;
    BottomColor := cl3DHiLight;
    PaintRect(1);
    TopColor := cl3DDkShadow;
    BottomColor := cl3DLight;
    PaintRect(1);
  end;
 end;

procedure TPhryListView.Paint;
var
  i, j :integer;
begin
  if csDesigning in ComponentState then
    with Canvas do
    begin
      Pen.Style := psDash;
      Brush.Style := bsClear;
      Rectangle(0, 0, Width, Height);
    end
  else
  begin
    i := 0;
    j := 0;
    if FTempBitmap.Height <> Height then
    begin
      i := Height - FTempBitmap.Height;
      FTempBitmap.Height := Height;
    end;
    if FTempBitmap.Width <> Width then
    begin
      j := Width - FTempBitmap.Width;
      FTempBitmap.Width := Width;
    end;
    if FWallPaper.Bitmap.Empty then
      FTempBitmap.Canvas.CopyRect(ClientRect, Canvas, ClientRect)
    else
    begin
      ControlStyle := ControlStyle + [csOpaque];
      FWallPaper.Paint(FTempBitmap.Canvas, ClientRect, Color);
      if i > 0 then
      begin
        Canvas.CopyRect(Rect(0,Height-i,Width,Height),FTempBitmap.Canvas,
                        Rect(0,Height-i,Width,Height));
      end;
      if j > 0 then
      begin
        Canvas.CopyRect(Rect(Width-j,0,Width,Height),FTempBitmap.Canvas,
                        Rect(Width-j,0,Width,Height));
      end;
    end;
    if FAutoLineHeight then
    begin
      FLineHeight := FTempBitmap.Canvas.TextHeight(' ');
      if Images <> nil then
        if Images.Height > FLineHeight then
          FLineHeight := Images.Height;
    end;
    TopMargin := 2;
    if FShowHeaders then
      TopMargin := TopMargin + FLineHeight + 4;
    FTempBitmap.Canvas.Brush.Color := Color;
    if not Transparent then
      FTempBitmap.Canvas.FillRect(ClientRect);
    if Items.Count * FLineHeight > Height - BottomMargin - TopMargin then
    begin
      FVScrollBar.Visible := True;
      FClientWidth := Width - LeftMargin - RightMargin - FVScrollBar.Width;
    end else
    begin
      FVScrollBar.Visible := False;
      FClientWidth := Width - LeftMargin - RightMargin;
    end;
    FClientHeight := Height - TopMargin - BottomMargin;
    FItemsShown := FClientHeight div FLineHeight;
    while (FItemOffset > 0) and (FItemsShown + FItemOffset > Items.Count) do
      Dec(FItemOffset);
    FVScrollBar.Position := FItemOffset + 1;
    FTempBitmap.Canvas.Font := Font;
    FTempBitmap.Canvas.Brush.Color := Color;
    FVScrollBar.Top := Top + 2;//TopMargin;
    FVScrollBar.Left := Left + Width - RightMargin - FVScrollBar.Width;
    FVScrollBar.Height := FClientHeight + FLineHeight + 4;
    if (FVScrollBar.Max <> Items.Count) and (Items.Count > 0) then
      FVScrollBar.Max := Items.Count;
    if FItemsShown <= FVScrollBar.Max then
      FVScrollBar.PageSize := FItemsShown
    else
      FVScrollBar.PageSize := FVScrollBar.Max;
    FVScrollBar.LargeChange := FVScrollBar.PageSize;
    if FVScrollBar.Parent = nil then
      FVScrollBar.Parent := Self.Parent;
    CalcColumnsWidth;
    if FShowHeaders then
    begin
      FTempBitmap.Canvas.Brush.Color := clBtnFace;
      FTempBitmap.Canvas.FillRect(Rect(LeftMargin, TopMargin - FLineHeight - 4, Width - RightMargin, TopMargin));
      for i := 0 to Columns.Count - 1 do
        PaintHeader(FTempBitmap.Canvas, i);
    end;
    FTempBitmap.Canvas.Brush.Style := bsClear;
    for i := FItemOffset to FItemOffset + (Height - 4) div FLineHeight do
      if i < Items.Count then
      begin
        for j := 0 to Columns.Count - 1 do
          PaintItem(Items[i], j);
      end;
    PaintBorder(FTempBitmap.Canvas, ClientRect, false);
    Canvas.CopyRect(ClientRect, FTempBitmap.Canvas, ClientRect);
  end;
end;

procedure TPhryListView.FindPos(Pos :TPoint; var item, column :Integer);
var
  i :integer;
begin
  item := -1;
  column := -1;
  if FLineHeight <= 0 then
    exit;
  Dec(Pos.y, TopMargin);
  Dec(Pos.x, LeftMargin);
  if Pos.y >= 0 then
    item := FItemOffset + Pos.y div FLineHeight;
  for i := 0 to Columns.Count - 1 do
    if (Columns.Items[i].Position <= Pos.x) and
       (Columns.Items[i].Position + Columns.Items[i].RealWidth >= Pos.x) then
       column := i;
end;

function TPhryListView.ItemRect(Item, Column :Integer) :TRect;
var
  r :TRect;
begin
  r := Rect(0,0,0,0);
  if (Columns.Count > 0) and (Column >=0) and (Column < Columns.Count) then
    if (Item >= FItemOffset - 1) and ((Item - FItemOffset) * FLineHeight < FClientHeight) then
    begin
      r.Top := (Item - FItemOffset) * FLineHeight + TopMargin;
      r.Bottom := r.Top + FLineHeight;// - BottomMargin;
      if Item < FItemOffset then
        Dec(r.Top, 4);
      r.Left := Columns.Items[Column].Position + LeftMargin;
      r.Right := r.Left + Columns.Items[Column].RealWidth;
      if Column = Columns.Count - 1 then
        r.Right := FClientWidth + LeftMargin;
    end;
  Result := r;
end;

procedure TPhryListView.SetImages(Value :TCustomImageList);
begin
  if Value <> FImages then
  begin
    FImages := Value;
    Refresh;
  end;
end;

procedure TPhryListView.PaintItem(Item :TPhryListItem; Column :Integer);
var
  PaintRect, r :TRect;
  imagex, imagey :Integer;
  textx, texty :Integer;
  s :string;
begin
  PaintRect := ItemRect(Item.Index, Column);
  r := PaintRect;
  if Item.Selected then
  begin
    FTempBitmap.Canvas.Brush.Color := clHighlight;
    FTempBitmap.Canvas.Brush.Style := bsSolid;
    FTempBitmap.Canvas.FillRect(PaintRect);
    FTempBitmap.Canvas.Brush.Style := bsClear;
  end;

  if Columns.Items[Column].OwnerDraw and Assigned(FOnDrawItem) then
    FOnDrawItem(FTempBitmap.Canvas, Item, Column, PaintRect)
  else
  begin
    if (Images <> nil) and (Item.ImageIndex >= 0) and (Column = 0) then
      if Item.ImageIndex < Images.Count then
      begin
        FImageBitmap.Height := PaintRect.Bottom - PaintRect.Top;
        FImageBitmap.Width := PaintRect.Right - PaintRect.Left;
        FImageBitmap.Canvas.FillRect(FImageBitmap.Canvas.ClipRect);
        imagex := PaintRect.Left + 2;
        imagey := (PaintRect.Bottom - PaintRect.Top - Images.Height) div 2;
        Images.Draw(FImageBitmap.Canvas, 0, imagey, Item.ImageIndex);
        FTempBitmap.Canvas.Draw(imagex, PaintRect.Top, FImageBitmap);
        PaintRect.Left := imagex + Images.Width;
      end;
    InflateRect(PaintRect, -2, 0);
    s := Item.Values[Column];
    if FTempBitmap.Canvas.TextWidth(s) > PaintRect.Right - PaintRect.Left then
    begin
      if Columns.Items[Column].Alignment = taRightJustify then
      begin
        s := '...' + s;
        while (FTempBitmap.Canvas.TextWidth(s) > PaintRect.Right - PaintRect.Left) and (Length(s)>3) do
          Delete(s, 4, 1);
      end else
      begin
        s := s + '...';
        while (FTempBitmap.Canvas.TextWidth(s) > PaintRect.Right - PaintRect.Left) and (Length(s)>3) do
          Delete(s, Length(s) - 3, 1);
      end;
    end;
    case Columns.Items[Column].Alignment of
      taLeftJustify: textx := PaintRect.Left;
      taRightJustify: textx := PaintRect.Right - FTempBitmap.Canvas.TextWidth(s);
      taCenter: textx := PaintRect.Left + (PaintRect.Right - PaintRect.Left - FTempBitmap.Canvas.TextWidth(s)) div 2;
      else textx := PaintRect.Left;
    end;
    texty := PaintRect.Top + (PaintRect.Bottom - PaintRect.Top - FTempBitmap.Canvas.TextHeight(s)) div 2;
    FTempBitmap.Canvas.Font := Canvas.Font;
    if Item.Selected then
      FTempBitmap.Canvas.Font.Color := Item.SelectedColor
    else
      FTempBitmap.Canvas.Font.Color := Item.Color;
    FTempBitmap.Canvas.TextRect(PaintRect, textx, texty, s);
  end;
end;

procedure TPhryListView.Scrolled(Sender :TObject);
begin
  if FVScrollBar.Position > FVScrollBar.Max - FVScrollBar.PageSize + 1 then
    FVScrollBar.Position := FVScrollBar.Max - FVScrollBar.PageSize + 1
  else
  begin
    FItemOffset := FVScrollBar.Position - 1;
    Refresh;
  end;
end;

procedure TPhryListView.CalcColumnsWidth;
var
  i :integer;
  TotalWidth :integer;
  ratio :Real;
begin
  TotalWidth := 0;
  for i := 0 to Columns.Count - 1 do
    inc(TotalWidth, Columns.Items[i].Width);
  ratio := FClientWidth / TotalWidth;
  TotalWidth := 0;
  for i := 0 to Columns.Count - 2 do
  begin
    Columns.Items[i].RealWidth := Round(Columns.Items[i].Width * ratio);
    Columns.Items[i].Position := TotalWidth;
    Inc(TotalWidth, Columns.Items[i].RealWidth);
  end;
  if Columns.Count > 0 then
  begin
    Columns.Items[Columns.Count - 1].Position := TotalWidth;
    Columns.Items[Columns.Count - 1].RealWidth := FClientWidth - TotalWidth;
  end;
end;

procedure TPhryListView.PaintHeader(ACanvas :TCanvas; Column :Integer);
var
  PaintRect :TRect;
  textx, texty, i :Integer;
  s :string;
begin
  PaintRect := ItemRect(FItemOffset - 1, Column);
  ACanvas.Brush.Color := clBtnFace;
  ACanvas.FillRect(PaintRect);
  InflateRect(PaintRect, -4, -2);
  s := Columns.Items[Column].Text;
  case Columns.Items[Column].Alignment of
//    taLeftJustify: textx := PaintRect.Left;
//    taRightJustify: textx := PaintRect.Right - FTempBitmap.Canvas.TextWidth(s);
    taCenter: textx := PaintRect.Left + (PaintRect.Right - PaintRect.Left - ACanvas.TextWidth(s)) div 2;
    else textx := PaintRect.Left + (PaintRect.Right - PaintRect.Left - ACanvas.TextWidth(s)) div 2;
  end;
  texty := PaintRect.Top + (PaintRect.Bottom - PaintRect.Top - ACanvas.TextHeight(s)) div 2;
  if Columns.Items[Column].Pressed then
  begin
    Inc(texty);
    Inc(textx);
  end;
  ACanvas.TextRect(PaintRect, textx, texty, s);
  if (Column = Items.SortColumn) and (Items.SortOrder <> soNone) then
  begin
    if Items.SortOrder = soAscending then
      i := 0
    else
      i := 9;
    inc(textx, FTempBitmap.Canvas.TextWidth(s) + 10);
    inc(texty, 3);
    ACanvas.CopyRect(Rect(textx, texty, textx+8, texty+8),FUpDownImage.Canvas,
                    Rect(i, 0, i+8, 8));
  end;
  InflateRect(PaintRect, 4, 2);
  PaintBorder(ACanvas, PaintRect, not Columns.Items[Column].Pressed);
end;

procedure TPhryListView.HandleMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  item, column :Integer;
  i :Integer;
begin
  FindPos(Point(X, Y), item, column);
  if Button = mbLeft then
  begin
    FClickedItem := item;
    FClickedColumn := column;
    if Cursor = crHSplit then
    begin
      FMouseMode := mmSplitting;
      for i := 0 to Columns.Count - 1 do
        Columns.Items[i].Width := Columns.Items[i].RealWidth;
      FPrevX := X;
    end else
    begin
      FMouseMode := mmClicking;
      if (item = -1) and (column >= 0) and (FShowHeaders) then
      begin
        Columns.Items[column].Pressed := True;
        PaintHeader(Canvas, column);
      end;

      if (item >= 0) and (item < Items.Count) then
      begin
        if (ssShift in Shift) and (FLastSelected >= 0) then
        begin
          for i := 0 to Items.Count - 1 do
            if i <> FLastSelected then
              Items.Items[i].Selected := False;
            if item > FLastSelected then
              for i := FLastSelected + 1 to item do
                Items.Items[i].Selected := True
            else
              for i := FLastSelected - 1 downto item do
                Items.Items[i].Selected := True;
        end else if ssCtrl in Shift then
        begin
          Items.Items[item].Selected := not Items.Items[item].Selected;
          FLastSelected := item;
        end else
        begin
          for i := 0 to Items.Count - 1 do
            Items.Items[i].Selected := False;
          Items.Items[item].Selected := True;
          FLastSelected := item;
        end;
        Refresh;
      end;

      if (item >= Items.Count) and (not (ssShift in Shift)) and (not (ssCtrl in Shift)) then
      begin
        for i := 0 to Items.Count - 1 do
          Items.Items[i].Selected := False;
        FLastSelected := -1;
        Refresh;
      end;
    end;
  end;
end;

procedure TPhryListView.HandleMouseUp(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  item, column :Integer;
begin
  FindPos(Point(X, Y), item, column);
  if Button = mbLeft then
  begin
    if (FMouseMode = mmClicking) and (FClickedItem = -1) and (FClickedColumn >= 0) and (FShowHeaders) then
    begin
      if (item = FClickedItem) and (column = FClickedColumn) then
      begin
        Columns.Items[column].Pressed := False;
        if Items.SortColumn = column then
          if Items.SortOrder = soAscending then
            Items.SortOrder := soDescending
          else
            Items.SortOrder := soAscending
        else
        begin
          Items.SortColumn := column;
          Items.SortOrder := soAscending;
        end;
        Items.Sort;
  //      PaintHeader(column);
        // create event
      end;
    end;

    FClickedItem := -1;
    FClickedColumn := -1;
    FMouseMode := mmNone;
  end;
end;

procedure TPhryListView.HandleMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  item, column :Integer;
  i, j, DeltaX :Integer;
begin
  FindPos(Point(X, Y), item, column);

  if FMouseMode = mmNone then
  begin
    FSplitMode := smNone;
    if (item = -1) and (column >= 0) and (FShowHeaders) then
    begin
      if LeftMargin + Columns.Items[column].Position +
         Columns.Items[column].RealWidth - X < 8 then
         if column < Columns.Count - 1 then
           FSplitMode := smRight;
      if X - Columns.Items[column].Position < 8 then
        if column > 0 then
          FSplitMode := smLeft;
    end;
    if FSplitMode <> smNone then
      Cursor := crHSplit
    else if Cursor <> crDefault then
      Cursor := crDefault;
  end;

  if (FMouseMode = mmClicking) and (FClickedItem = -1) and (FClickedColumn >= 0) then
  begin
    if (item <> FClickedItem) or (column <> FClickedColumn) then
    begin
      if Columns.Items[FClickedColumn].Pressed then
      begin
        Columns.Items[FClickedColumn].Pressed := False;
        PaintHeader(Canvas, FClickedColumn);
      end
    end else
    begin
      if not Columns.Items[FClickedColumn].Pressed then
      begin
        Columns.Items[FClickedColumn].Pressed := True;
        PaintHeader(Canvas, FClickedColumn);
      end;
    end;
  end;

  if (FMouseMode = mmSplitting) and (X >= LeftMargin) and (X < Width - RightMargin) then
  begin
    if FSplitMode = smLeft then
    begin
      DeltaX := X - FPrevX;
      if DeltaX < 0 then
      begin
        j := 0;
        for i := 0 to FClickedColumn - 1 do
          inc(j, Columns.Items[i].Width);
        if j + DeltaX < MinColumnWidth * FClickedColumn then
          DeltaX := MinColumnWidth * FClickedColumn - j;
        DeltaX := abs(DeltaX);
        FPrevX := FPrevX - DeltaX;
        Columns.Items[FClickedColumn].Width := Columns.Items[FClickedColumn].Width + DeltaX;
        while DeltaX > 0 do
        begin
          j := DeltaX div FClickedColumn;
          if j = 0 then
            j := 1;
          for i := 0 to FClickedColumn - 1 do
          begin
            if Columns.Items[i].Width - j < MinColumnWidth then
            begin
              dec(DeltaX, Columns.Items[i].Width - MinColumnWidth);
              Columns.Items[i].Width := MinColumnWidth;
            end else
            begin
              dec(DeltaX, j);
              Columns.Items[i].Width := Columns.Items[i].Width - j;
            end;
            if DeltaX = 0 then
              j := 0;
          end;
        end;
      end else if DeltaX > 0 then
      begin
        if Columns.Items[FClickedColumn].Width - DeltaX < MinColumnWidth then
          DeltaX := Columns.Items[FClickedColumn].Width - MinColumnWidth;
        FPrevX := FPrevX + DeltaX;
        Columns.Items[FClickedColumn].Width := Columns.Items[FClickedColumn].Width - DeltaX;
        while DeltaX > 0 do
        begin
          j := DeltaX div FClickedColumn;
          if j = 0 then
            j := 1;
          for i := 0 to FClickedColumn - 1 do
          begin
            dec(DeltaX, j);
            Columns.Items[i].Width := Columns.Items[i].Width + j;
            if DeltaX = 0 then
              j := 0;
          end;
        end;
      end;
    end else
    begin
      DeltaX := X - FPrevX;
      if DeltaX > 0 then
      begin
        j := 0;
        for i := FClickedColumn + 1 to Columns.Count - 1 do
          inc(j, Columns.Items[i].Width);
        if j - DeltaX < MinColumnWidth * (Columns.Count - FClickedColumn - 1) then
          DeltaX := j - MinColumnWidth * (Columns.Count - FClickedColumn - 1);
        FPrevX := FPrevX + DeltaX;
        Columns.Items[FClickedColumn].Width := Columns.Items[FClickedColumn].Width + DeltaX;
        while DeltaX > 0 do
        begin
          j := DeltaX div (Columns.Count - FClickedColumn - 1);
          if j = 0 then
            j := 1;
          for i := FClickedColumn + 1 to Columns.Count - 1 do
          begin
            if Columns.Items[i].Width - j < MinColumnWidth then
            begin
              dec(DeltaX, Columns.Items[i].Width - MinColumnWidth);
              Columns.Items[i].Width := MinColumnWidth;
            end else
            begin
              dec(DeltaX, j);
              Columns.Items[i].Width := Columns.Items[i].Width - j;
            end;
            if DeltaX = 0 then
              j := 0;
          end;
        end;
      end else if DeltaX < 0 then
      begin
        if Columns.Items[FClickedColumn].Width + DeltaX < MinColumnWidth then
          DeltaX := Columns.Items[FClickedColumn].Width - MinColumnWidth;
        DeltaX := abs(DeltaX);
        FPrevX := FPrevX - DeltaX;
        Columns.Items[FClickedColumn].Width := Columns.Items[FClickedColumn].Width - DeltaX;
        while DeltaX > 0 do
        begin
          j := DeltaX div (Columns.Count - FClickedColumn - 1);
          if j = 0 then
            j := 1;
          for i := FClickedColumn + 1 to Columns.Count - 1 do
          begin
            dec(DeltaX, j);
            Columns.Items[i].Width := Columns.Items[i].Width + j;
            if DeltaX = 0 then
              j := 0;
          end;
        end;
      end;
    end;
    Refresh;
  end;
end;

procedure TPhryListView.HandleContextPopup(Sender: TObject; MousePos: TPoint; var Handled: Boolean);
var
  item, column :Integer;
begin
  if Assigned(FOnPopup) then
  begin                        
    FindPos(MousePos, item, column);
    if (item < Items.Count) and (Item >= 0) then
      FOnPopup(Sender, Items[item], column, Handled)
    else if (item >= Items.Count) then
      FOnPopup(Sender, nil, -1, Handled)
    else
      FOnPopup(Sender, nil, column, Handled);
  end;
end;

procedure Register;
begin
  RegisterComponents('Phry', [TPhryListView]);
end;
end.
